% Copyright (c) 2016 Giampaolo D'Alessandro
% Mathematical Sciences - University of Southampton
% Email: dales@soton.ac.uk
% 
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files (the "Software"), to deal
% in the Software without restriction, including without limitation the rights
% to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
% copies of the Software, and to permit persons to whom the Software is
% furnished to do so, subject to the following conditions:
% 
% The above copyright notice and this permission notice shall be included in
% all copies or substantial portions of the Software.
% 
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
% OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
% SOFTWARE.

% Script to fit the refractive index parameters

% Data from Li et al, IEEE J. Disp. Tech 1(1), 51 (2005)

% TL216 at $\lambda = 589$~nm
% ne = [1.7408 1.7365 1.7315 1.7259 1.7211 1.7156 1.7093 1.7030 1.6957];
% no = [1.5246 1.5240 1.5228 1.5218 1.5209 1.5204 1.5198 1.5193 1.5196];
% T = 15:5:55;
% Td = 331.35;  % Clearing temperature

% E7 data at $\lambda=633$~nm
ne = [1.7446 1.7378 1.7305 1.7229 1.7142 1.7037 1.6919 1.6761 1.644];
no = [1.5197 1.5188 1.5189 1.5189 1.5191 1.5205 1.52221 1.526 1.5377];
T = 15:5:55;
Td = 331.35;  % Clearing temperature

% % E7 data at $\lambda=589$~nm
% ne = [1.7542 1.7466 1.7394 1.7317 1.7231 1.7124 1.7001 1.6837 1.6511];
% no = [1.523 1.5227 1.5225 1.5226 1.5231 1.5239 1.5261 1.5299 1.5428];
% T = 15:5:55;
% Td = 331.35;  % Clearing temperature

% E7 data at $\lambda=546$~nm
% ne = [1.7664 1.7589 1.7512 1.7433 1.7344 1.7237 1.7109 1.6941 1.6601];
% no = [1.5273 1.5269 1.5268 1.5271 1.5277 1.5287 1.5308 1.535 1.5479];
% T = 15:5:55;
% Td = 331.35;  % Clearing temperature

% E7 data at $\lambda=486$~nm
% ne = [1.7921 1.7847 1.7763 1.7679 1.7581 1.7472 1.7333 1.7154 1.6779];
% no = [1.536 1.5357 1.5353 1.5357 1.5369 1.5383 1.5406 1.5449 1.5588];
% T = 15:5:55;
% Td = 331.35;  % Clearing temperature

np = length(ne);
n_av = (ne+2*no)/3;
T_K = 273.15;
T_A = 293.15;

% Express the temperatures of the measurements in Kelvin
T = T + T_K;

% Scalar order parameter (all temperatures must be in Kelvin)
S = @(T,beta,Td) (1-T/Td).^beta;
% Fitting functions (all temperatures must be in Kelvin)
N_e = @(T,A,B,Dn,beta,Td) A - B*T + 2*Dn*S(T,beta,Td)/3;
N_o = @(T,A,B,Dn,beta,Td) A - B*T - Dn*S(T,beta,Td)/3;
N_av = @(T,A,B) A - B*T;
dN_edT = @(T,A,B,Dn,beta,Td) -B - 2*Dn*beta*S(T,beta,Td)./(3*(Td-T));
dN_odT = @(T,A,B,Dn,beta,Td) -B + Dn*beta*S(T,beta,Td)./(3*(Td-T));

% Distance function when fitting the average
D_av = @(P_av) sum((n_av-P_av(1)+P_av(2)*T).^2);
% Distance function when using the clearing temperature in the the Heller
% approximation.
D_Td = @(P,A,B,Td) sum((ne-N_e(T,A,B,P(1),P(2),Td)).^2 + ...
                    (no-N_o(T,A,B,P(1),P(2),Td)).^2);
% Distance function when leaving the clearing temperature in the the Heller
% approximation as a free parameter.
D = @(P,A,B) sum((ne-N_e(T,A,B,P(1),P(2),P(3))).^2 + ...
                    (no-N_o(T,A,B,P(1),P(2),P(3))).^2);
                  
% Find minimum of D_av using the standard linear regressin formula
P_av = [(mean(T.^2)*mean(n_av) - mean(T)*mean(n_av.*T))/...
        (mean(T.^2)-mean(T)^2), ...
        (mean(T)*mean(n_av) - mean(n_av.*T))/...
        (mean(T.^2)-mean(T)^2)];
display(sprintf('A=%.4f; B=%.4e; D_min = %.3e', ...
  P_av(1),P_av(2),D_av(P_av)));

%Now find minimum of D_td as a function of $\Delta n_0$ and $\beta$.

P_Td_0 = [0.2,0.24];
[P_Td,D_Td_min] = fminunc(@(P) D_Td(P,P_av(1),P_av(2),Td),P_Td_0);
display(sprintf('Delta n_0=%.4f; beta=%.4f; Min. Distance=%.3e', ...
  P_Td(1),P_Td(2),D_Td_min));

% Now find minimum of D as a function of $\Delta n_0$, $\beta$ and
% $T^{\dagger}$. 

P_0 = [0.2,0.24,Td]; 
[P,D_min] = fminunc(@(P) D(P,P_av(1),P_av(2)),P_0);
display(sprintf('Delta n_0=%.4f; beta=%.4f; Td=%.2f; Min. Distance=%.3e', ...
  P(1),P(2),P(3),D_min));

% Values obtained
% TL216 - Same as Li et al (2005)
% E7
% A=1.7484; B=5.3362e-04;
% Delta n_0=0.3523; beta=0.2161; Td=329.28; Min. Distance=7.602e-07

% Plot the results
FontName='Times';
FontSize=32;
FontAngle='normal'; %'italic';
LineWidth=2;
MarkerSize=12;  % In points (72 points per inch)

% figure(1);
% clf;
% set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
%   'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
% axes('Parent',gcf,...
%   'YMinorTick','on','XMinorTick','on',...
%   'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
% plot(T-T_K,ne,'r-o',T-T_K,no,'b-d', ...
%   T-T_K,N_e(T,P_av(1),P_av(2),P_Td(1),P_Td(2),Td),'r--',...
%   T-T_K,N_o(T,P_av(1),P_av(2),P_Td(1),P_Td(2),Td),'b--', ...
%   'LineWidth',LineWidth,'MarkerSize',MarkerSize);
% xlabel('T [C]');
% ylabel('n_e and n_o');
% legend(gca,'n_e Exp','n_o Exp','n_e Fit','n_o Fit','Location','West');

% figure(2);
% clf;
% set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
%   'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
% axes('Parent',gcf,...
%   'YMinorTick','on','XMinorTick','on',...
%   'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
% plot(T-T_K,ne,'r-o',T-T_K,no,'b-d', ...
%   T-T_K,N_e(T,P_av(1),P_av(2),P(1),P(2),P(3)),'r--',...
%   T-T_K,N_o(T,P_av(1),P_av(2),P(1),P(2),P(3)),'b--', ...
%   'LineWidth',LineWidth,'MarkerSize',MarkerSize);
% xlabel('T');
% ylabel('n_e and n_o');
% legend(gca,'n_e Exp','n_o Exp','n_e Fit','n_o Fit','Location','West');
% saveas(gcf,'RefractiveIndex_Temperature','pdf');

figure(3);
clf;
set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
  'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
axes('Parent',gcf,...
  'YMinorTick','on','XMinorTick','on',...
  'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
plot(T-T_K,dN_edT(T,P_av(1),P_av(2),P(1),P(2),P(3)),'r-o',...
  T-T_K,dN_odT(T,P_av(1),P_av(2),P(1),P(2),P(3)),'b-d', ...
  'LineWidth',LineWidth,'MarkerSize',MarkerSize);
hold on

xlabel('T [C]');
ylabel('d n_e/dt and d n_o/dt');
legend(gca,'d n_e/dT','d n_o/dT','Location','SouthWest');
saveas(gcf,'dndT','pdf');

T_total = [297.3033 298.25866 299.44625 300.713 301.8606 302.78242 304.17626];         %Temperature of the system
dN_edT_alpha = dN_edT(T_total,P_av(1),P_av(2),P(1),P(2),P(3));                         %Corresponding dn_edT
delT_mod = [293.4393 293.50652 293.59089 293.68211 293.76667 293.84522 293.94]-T_A;    %Maximum amplitude of modulation
deln = dN_edT_alpha.*(delT_mod/2);                                                     %Change in refractive index seen by the probe beam 
n2 = deln/19.5;                                                                        %n2 coefficient in cm^2/W    
alphaexp = [42.16 53.25 67.95 84.9 101.6 116.08 140.13];
n2exp = [0 1 1.5 2.1 2.35 2.9 3.1]*-1e-5;


% figure(4);
% clf;
% set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
%   'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
% axes('Parent',gcf,...
%   'YMinorTick','on','XMinorTick','on',...
%   'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
% plot(alphaexp,n2,'r-o',...
%   alphaexp,n2exp,'bd', ...
%   'LineWidth',LineWidth,'MarkerSize',MarkerSize);
% hold on
% xlabel('\alpha [cm^{-1}]');
% ylabel('n_{2} [cm^{2}/W]');
% legend(gca,'Theoretical','Experimental','Location','SouthWest');
% saveas(gcf,'varying_alpha','pdf');